<?php
session_start();
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once 'config.php';
require_once 'rut.php';

function e($text) { return htmlspecialchars($text ?? '', ENT_QUOTES, 'UTF-8'); }
function user() { return $_SESSION['user'] ?? null; }
function fix_month($month) {
    // Asegura que el formato YYYY-MM se convierta a YYYY-MM-01
    if (preg_match('/^(\d{4})-(\d{2})$/', $month, $m)) {
        return $m[1] . '-' . $m[2] . '-01';
    }
    return $month;
}


$user = user();
// Control de Acceso: Solo Rol 1 (Superadmin) y Rol 2 (Admin SSR)
if (!$user || $user['rol'] > 2) {
    header('Location: index.php');
    exit;
}

$config = include 'config.php';
$dsn = "mysql:host={$config['db']['host']};dbname={$config['db']['dbname']};charset=utf8mb4";
$pdo = new PDO($dsn, $config['db']['user'], $config['db']['pass'], [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
]);

$success = $error = '';
$rendicion_id = (int)($_GET['id'] ?? 0);
$rendicion = null;

if ($rendicion_id > 0) {
    // Cargar la rendición, incluyendo el nombre de la SSR
    $stmt = $pdo->prepare("
        SELECT r.*, s.nombre as ssr_nombre 
        FROM rendiciones r 
        JOIN ssr s ON r.ssr_id = s.id 
        WHERE r.id = ?
    ");
    $stmt->execute([$rendicion_id]);
    $rendicion = $stmt->fetch();

    if (!$rendicion) {
        $error = "❌ Rendición ID #{$rendicion_id} no encontrada.";
    } 
    // CONTROL DE ACCESO CRÍTICO (VISUALIZACIÓN): Rol 2 solo edita rendiciones de su SSR
    elseif ($user['rol'] == 2 && $rendicion['ssr_id'] != $user['ssr_id']) {
        $error = "❌ No tiene permisos para editar esta rendición. Pertenece a otra SSR.";
        $rendicion = null; // Bloquear la visualización
    }
} else {
    $error = "❌ ID de rendición no especificado.";
}


// ===============================================
// PROCESAMIENTO POST (Guardar Edición)
// ===============================================

if ($_SERVER['REQUEST_METHOD'] == 'POST' && $rendicion) { // Solo procesamos si hay una rendición cargada y autorizada
    
    // Re-validar el permiso en el POST
    $current_rendicion_id = (int)$_POST['rendicion_id'];
    $stmt_check = $pdo->prepare("SELECT ssr_id FROM rendiciones WHERE id = ?");
    $stmt_check->execute([$current_rendicion_id]);
    $check_rendicion = $stmt_check->fetch();
    
    // Si la rendición existe y el usuario es Rol 2, verificamos si es de su SSR
    if (!$check_rendicion || ($user['rol'] == 2 && $check_rendicion['ssr_id'] != $user['ssr_id'])) {
        $error = "❌ Intento de edición fallido. Permiso denegado.";
        // No continuar con el procesamiento del POST
    } else {
        // Recoger datos
        $titulo = trim($_POST['titulo'] ?? '');
        $tipo_periodo = trim($_POST['tipo_periodo'] ?? '');
        $periodo_inicio = fix_month(trim($_POST['periodo_inicio'] ?? ''));
        $periodo_fin = fix_month(trim($_POST['periodo_fin'] ?? ''));
        $estado = trim($_POST['estado'] ?? 'borrador');
        $contenido = trim($_POST['contenido'] ?? '');
        
        // El SSR_ID solo puede ser modificado por el Superadmin (Rol 1)
        $ssr_id_update = $check_rendicion['ssr_id']; // Por defecto, usamos el original
        if ($user['rol'] == 1) { 
            $ssr_id_update = (int)($_POST['ssr_id'] ?? $check_rendicion['ssr_id']);
        }
        
        if (empty($titulo) || empty($periodo_inicio) || empty($periodo_fin)) {
            $error = "❌ Todos los campos marcados con * son obligatorios.";
        }

        if (!$error) {
            try {
                $sql = "UPDATE rendiciones SET 
                            titulo = ?, 
                            ssr_id = ?, 
                            tipo_periodo = ?, 
                            periodo_inicio = ?, 
                            periodo_fin = ?, 
                            estado = ?, 
                            contenido = ?
                        WHERE id = ?";
                
                $pdo->prepare($sql)->execute([
                    $titulo, 
                    $ssr_id_update, 
                    $tipo_periodo, 
                    $periodo_inicio, 
                    $periodo_fin, 
                    $estado, 
                    $contenido,
                    $current_rendicion_id
                ]);

                $success = "✅ Rendición '<strong>".e($titulo)."</strong>' actualizada exitosamente.";
                
                // Recargar para mostrar los datos actualizados
                header("Location: editar_rendicion.php?id={$current_rendicion_id}&msg=success");
                exit;

            } catch (Exception $e) {
                $error = "❌ Error al actualizar la rendición: " . $e->getMessage();
            }
        }
    }
}

// Manejar el mensaje de redirección
if (isset($_GET['msg']) && $_GET['msg'] === 'success') {
    $success = "✅ Rendición actualizada exitosamente.";
    // Recargar la data de la rendición para la vista después de la actualización
    $stmt = $pdo->prepare("SELECT r.*, s.nombre as ssr_nombre FROM rendiciones r JOIN ssr s ON r.ssr_id = s.id WHERE r.id = ?");
    $stmt->execute([$rendicion_id]);
    $rendicion = $stmt->fetch();
}


// Cargar la lista de SSRs para el selector (solo si es Superadmin)
$ssrs = [];
if ($user['rol'] == 1) {
    $stmt_ssrs = $pdo->query("SELECT id, nombre FROM ssr WHERE activo = 1 ORDER BY nombre");
    $ssrs = $stmt_ssrs->fetchAll();
}

// Si la rendición no se pudo cargar o el acceso fue denegado, $rendicion será null.

?>
<!DOCTYPE html>
<html>
<head>
    <title>✏️ Editar Rendición</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        :root {
            --primary: #77216F;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --info: #17a2b8;
        }
        body{font-family:'Segoe UI',Arial,sans-serif;background:#f8f9fa;padding:30px;min-height:100vh;}
        .container{max-width:800px;margin:0 auto;}
        .header{background:linear-gradient(135deg,var(--primary),#5E2750);color:white;padding:30px;border-radius:15px;text-align:center;margin-bottom:30px;}
        .form-container{background:white;padding:40px;border-radius:15px;box-shadow:0 10px 30px rgba(0,0,0,0.1);}
        .form-group{margin-bottom:20px;}
        label{display:block;margin-bottom:8px;color:#333;font-weight:600;}
        input[type="text"], input[type="month"], select, textarea{padding:12px;border:1px solid #ddd;border-radius:8px;width:100%;font-size:16px;font-family:inherit;}
        textarea{resize:vertical;min-height:150px;}
        .btn{padding:12px 25px;border-radius:8px;font-weight:600;cursor:pointer;transition:all 0.3s;text-decoration:none;display:inline-block;border:none;}
        .btn-warning{background:var(--warning);color:#333;}
        .btn-warning:hover{background:#e0a800;}
        .btn-back{background:#6c757d;color:white;}
        .btn-back:hover{background:#5a6268;}
        .alert-error{background:#f8d7da;color:var(--danger);padding:15px;border-radius:8px;margin-bottom:20px;border-left:5px solid var(--danger);}
        .alert-success{background:#d4edda;color:var(--success);padding:15px;border-radius:8px;margin-bottom:20px;border-left:5px solid var(--success);}
    </style>
</head>
<body>
<div class="container">
    <div class="header">
        <h1><i class="fas fa-edit"></i> Editar Rendición</h1>
        <p>Usuario: **<?= e($user['nombre_completo']) ?>** (<?= $user['rol'] == 1 ? 'Superadmin' : 'Admin SSR' ?>)</p>
    </div>

    <?php if ($error): ?><div class="alert-error"><?= $error ?></div><?php endif; ?>
    <?php if ($success): ?><div class="alert-success"><?= $success ?></div><?php endif; ?>

    <?php if ($rendicion): ?>
    <div class="form-container">
        <h2 style="margin-bottom: 20px;">Editando: **<?= e($rendicion['titulo']) ?>** (#<?= $rendicion['id'] ?>)</h2>
        <p style="color:#666;margin-bottom:30px;">SSR Actual: **<?= e($rendicion['ssr_nombre']) ?>**</p>
        
        <form method="POST">
            <input type="hidden" name="rendicion_id" value="<?= $rendicion['id'] ?>">

            <div class="form-group">
                <label>Título *</label>
                <input type="text" name="titulo" value="<?= e($_POST['titulo'] ?? $rendicion['titulo']) ?>" required maxlength="255">
            </div>

            <div class="form-group">
                <label>SSR *</label>
                <?php if ($user['rol'] == 1): // Superadmin puede cambiar la SSR ?>
                <select name="ssr_id" required>
                    <?php foreach ($ssrs as $s): ?>
                        <option value="<?= $s['id'] ?>" <?= $s['id'] == ($_POST['ssr_id'] ?? $rendicion['ssr_id']) ? 'selected' : '' ?>>
                            <?= e($s['nombre']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <?php else: // Admin SSR solo puede ver la suya (no editable) ?>
                <input type="text" value="<?= e($rendicion['ssr_nombre']) ?>" disabled>
                <input type="hidden" name="ssr_id" value="<?= $rendicion['ssr_id'] ?>">
                <?php endif; ?>
            </div>

            <div class="form-group">
                <label>Tipo de Período *</label>
                <select name="tipo_periodo" required>
                    <?php 
                    $selected_tipo = $_POST['tipo_periodo'] ?? $rendicion['tipo_periodo'];
                    $tipos = ['mensual' => '📅 Mensual', 'trimestral' => '📊 Trimestral', 'semestral' => '📈 Semestral', 'anual' => '📉 Anual'];
                    foreach($tipos as $val => $text):
                    ?>
                        <option value="<?= $val ?>" <?= $val == $selected_tipo ? 'selected' : '' ?>><?= $text ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div style="display:grid;grid-template-columns:1fr 1fr;gap:15px;">
                <div class="form-group">
                    <label>Desde * (YYYY-MM)</label>
                    <input type="month" name="periodo_inicio" value="<?= e(substr($_POST['periodo_inicio'] ?? $rendicion['periodo_inicio'], 0, 7)) ?>" required>
                </div>
                <div class="form-group">
                    <label>Hasta * (YYYY-MM)</label>
                    <input type="month" name="periodo_fin" value="<?= e(substr($_POST['periodo_fin'] ?? $rendicion['periodo_fin'], 0, 7)) ?>" required>
                </div>
            </div>

            <div class="form-group">
                <label>Estado *</label>
                <select name="estado" required>
                    <option value="borrador" <?= ($_POST['estado'] ?? $rendicion['estado']) == 'borrador' ? 'selected' : '' ?>>📝 Borrador</option>
                    <option value="publicada" <?= ($_POST['estado'] ?? $rendicion['estado']) == 'publicada' ? 'selected' : '' ?>>🟢 Publicada</option>
                </select>
            </div>

            <div class="form-group">
                <label>Contenido / Descripción</label>
                <textarea name="contenido" placeholder="Detalles de la rendición..."><?= e($_POST['contenido'] ?? $rendicion['contenido']) ?></textarea>
            </div>
            
            <div style="margin-top:30px; text-align: right;">
                <a href="rendiciones_admin.php" class="btn btn-back"><i class="fas fa-arrow-left"></i> Volver</a>
                <button type="submit" class="btn btn-warning"><i class="fas fa-save"></i> Guardar Cambios</button>
            </div>
        </form>
    </div>
    <?php else: ?>
    <div class="form-container" style="text-align:center;padding:60px;">
        <h3>⚠️ Acceso Denegado</h3>
        <p style="color:var(--danger); font-size: 1.1em;"><?= e($error) ?></p>
        <p>Verifique si la rendición existe o si tiene permisos de SSR para editarla.</p>
        <a href="rendiciones_admin.php" class="btn btn-back" style="margin-top:20px;"><i class="fas fa-arrow-left"></i> Volver a la Lista</a>
    </div>
    <?php endif; ?>
</div>
</body>
</html>