<?php
session_start();
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once 'config.php';
require_once 'rut.php';

function e($text) { return htmlspecialchars($text ?? '', ENT_QUOTES, 'UTF-8'); }
function user() { return $_SESSION['user'] ?? null; }

$user = user();
if (!$user) {
    header('Location: index.php');
    exit;
}

$config = include 'config.php';
$dsn = "mysql:host={$config['db']['host']};dbname={$config['db']['dbname']};charset=utf8mb4";
$pdo = new PDO($dsn, $config['db']['user'], $config['db']['pass'], [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
]);

$success = $error = '';
$rendicion_id = (int)($_GET['id'] ?? 0);

if (!$rendicion_id) {
    header('Location: index.php');
    exit;
}

// ===============================================
// 1. CARGAR Y VALIDAR RENDICIÓN (Acceso por SSR)
// ===============================================

$stmt = $pdo->prepare("
    SELECT r.*, s.nombre as ssr_nombre 
    FROM rendiciones r 
    JOIN ssr s ON r.ssr_id = s.id 
    WHERE r.id = ?
");
$stmt->execute([$rendicion_id]);
$rendicion = $stmt->fetch();

$can_admin_interact = false; // Permiso para Rol 1/2 para responder
$can_socio_access = false; // Permiso para Rol 5 para ver y preguntar
$access_error = '';

if (!$rendicion) {
    $access_error = "❌ Rendición ID #{$rendicion_id} no encontrada.";
} else {
    // --- Lógica de acceso de ADMINISTRACIÓN (Rol 1 y 2) ---
    if ($user['rol'] == 1) { // Superadmin
        $can_admin_interact = true;
    } elseif ($user['rol'] == 2) { // Admin SSR
        if ($rendicion['ssr_id'] == $user['ssr_id']) {
            $can_admin_interact = true;
        } else {
            $access_error = "❌ No tiene permisos para administrar preguntas de esta rendición (SSR no asignada).";
        }
    }
    
    // --- Lógica de acceso SOCIO (Rol 5) ---
    if ($user['rol'] == 5) {
        if ($rendicion['ssr_id'] == $user['ssr_id'] && $rendicion['estado'] == 'publicada') {
            $can_socio_access = true;
        } else {
            // Si es socio y no cumple, lo redirigimos a su lista.
            header('Location: rendiciones_socios.php');
            exit;
        }
    }

    // Si el usuario es administrador y no tiene permisos
    if ($user['rol'] <= 2 && !$can_admin_interact) {
        $rendicion = null; // Bloquear la visualización del foro
    }
}


// ===============================================
// 2. PROCESAMIENTO POST (Hacer Pregunta - Rol 5)
// ===============================================

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['hacer_pregunta'])) {
    if ($user['rol'] == 5 && $can_socio_access) {
        $pregunta = trim($_POST['pregunta'] ?? '');
        if (empty($pregunta)) {
            $error = "❌ El campo de pregunta no puede estar vacío.";
        } else {
            $sql = "INSERT INTO preguntas_rendiciones (rendicion_id, socio_id, pregunta) VALUES (?, ?, ?)";
            $pdo->prepare($sql)->execute([$rendicion_id, $user['id'], $pregunta]);
            
            // Redirigir para evitar re-envío
            header("Location: foro_rendicion.php?id={$rendicion_id}&msg=question_success");
            exit;
        }
    }
}

// ===============================================
// 3. PROCESAMIENTO POST (Responder Pregunta - Rol 1/2)
// ===============================================

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['responder'])) {
    // Solo permitimos responder a Rol 1 o 2 que tengan acceso
    if ($user['rol'] <= 2 && $can_admin_interact) {
        $pregunta_id = (int)($_POST['pregunta_id'] ?? 0);
        $respuesta = trim($_POST['respuesta'] ?? '');

        if (empty($respuesta)) {
            $error = "❌ El campo de respuesta no puede estar vacío.";
        }

        if (!$error) {
            // Asegurarse de que la pregunta pertenece a esta rendición
            $stmt_q = $pdo->prepare("SELECT rendicion_id FROM preguntas_rendiciones WHERE id = ?");
            $stmt_q->execute([$pregunta_id]);
            $q = $stmt_q->fetch();

            if (!$q || $q['rendicion_id'] != $rendicion_id) {
                $error = "❌ La pregunta no existe o no pertenece a esta rendición.";
            } else {
                // Actualizar la pregunta
                $sql = "UPDATE preguntas_rendiciones SET respuesta = ?, respondido = 1, respondido_por = ?, updated_at = NOW() WHERE id = ?";
                $pdo->prepare($sql)->execute([$respuesta, $user['id'], $pregunta_id]);
                
                // Redirigir para evitar re-envío y mostrar el mensaje
                header("Location: foro_rendicion.php?id={$rendicion_id}&msg=answer_success#q{$pregunta_id}");
                exit;
            }
        }
    } else {
        $error = $access_error ?: "❌ No tiene permisos para responder preguntas.";
    }
}


// Manejar mensajes de redirección
if (isset($_GET['msg'])) {
    if ($_GET['msg'] === 'question_success') {
        $success = "✅ Pregunta enviada exitosamente. Espere la respuesta del administrador.";
    } elseif ($_GET['msg'] === 'answer_success') {
        $success = "✅ Respuesta publicada exitosamente.";
    }
}

// ===============================================
// 4. CARGAR PREGUNTAS
// ===============================================

$preguntas = [];
if ($rendicion) { // Solo cargamos si el usuario tiene permiso para ver la rendición
    $sql_preguntas = "
        SELECT 
            p.*, 
            s_pregunta.nombre_completo as socio_pregunta,
            s_respuesta.nombre_completo as admin_respuesta
        FROM preguntas_rendiciones p
        JOIN socios s_pregunta ON p.socio_id = s_pregunta.id
        LEFT JOIN socios s_respuesta ON p.respondido_por = s_respuesta.id
        WHERE p.rendicion_id = ?
        ORDER BY p.created_at DESC
    ";
    $stmt_preguntas = $pdo->prepare($sql_preguntas);
    $stmt_preguntas->execute([$rendicion_id]);
    $preguntas = $stmt_preguntas->fetchAll();
}

// Botón de Volver
$back_link = ($user['rol'] == 5) ? 'rendiciones_socios.php' : 'rendiciones_admin.php';
?>
<!DOCTYPE html>
<html>
<head>
    <title>🗣️ Foro Rendición #<?= $rendicion_id ?></title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        :root {
            --primary: #77216F;
            --secondary: #5E2750;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --info: #17a2b8;
            --light: #f8f9fa;
        }
        body{font-family:'Segoe UI',Arial,sans-serif;background:#f8f9fa;padding:30px;min-height:100vh;}
        .container{max-width:900px;margin:0 auto;}
        .header{background:linear-gradient(135deg,var(--primary),var(--secondary));color:white;padding:30px;border-radius:15px;text-align:center;margin-bottom:30px;}
        .alert-error{background:#f8d7da;color:var(--danger);padding:15px;border-radius:8px;margin-bottom:20px;border-left:5px solid var(--danger);}
        .alert-success{background:#d4edda;color:var(--success);padding:15px;border-radius:8px;margin-bottom:20px;border-left:5px solid var(--success);}
        
        .rendicion-info{background:white;padding:25px;border-radius:12px;box-shadow:0 4px 15px rgba(0,0,0,0.1);margin-bottom:30px;border-left:5px solid var(--info);}
        .rendicion-info h2{color:var(--primary);margin-top:0;}
        .rendicion-info p{margin-bottom:5px;}
        .rendicion-info a{color:var(--info);font-weight:600;}

        .pregunta-form, .no-access{background:white;padding:25px;border-radius:12px;box-shadow:0 4px 15px rgba(0,0,0,0.05);margin-bottom:30px;border-top:3px solid var(--success);}
        .pregunta-form textarea{width:100%;padding:10px;border:1px solid #ddd;border-radius:6px;margin-bottom:10px;resize:vertical;}
        
        .pregunta{margin-bottom:25px;padding:25px;background:var(--light);border-radius:12px;border-left:5px solid var(--primary);position:relative;}
        .pregunta-meta{font-size:13px;color:#999;margin-bottom:10px;}
        .pregunta h4{color:var(--primary);margin-bottom:10px;}
        .respuesta{margin-top:15px;padding:15px;background:var(--info);color:white;border-radius:0 8px 8px 0;border-left:4px solid var(--success);font-weight:500;}
        .respuesta strong{display:block;font-size:13px;font-weight:400;color:#e6f7ff;margin-bottom:5px;}
        .no-respuesta{color:#999;padding:15px;background:#fff3cd;border-left:4px solid var(--warning);border-radius:0 8px 8px 0;margin-top:15px;}
        
        .btn{padding:8px 15px;border-radius:6px;font-weight:600;cursor:pointer;transition:all 0.3s;text-decoration:none;display:inline-block;border:none;}
        .btn-responder{background:var(--success);color:white;}
        .btn-responder:hover{background:#218838;}
        .btn-send{background:var(--primary);color:white;}
        .btn-back{background:#6c757d;color:white;}
        
        .respuesta-form{margin-top:10px;padding:15px;border:1px solid #ddd;border-radius:8px;background:#fefefe;}
    </style>
</head>
<body>
<div class="container">
    <div class="header">
        <h1><i class="fas fa-comments"></i> Foro de Rendición</h1>
        <p>Rendición ID #<?= $rendicion_id ?> | Usuario: **<?= e($user['nombre_completo']) ?>**</p>
    </div>

    <?php if ($error): ?><div class="alert-error"><?= $error ?></div><?php endif; ?>
    <?php if ($success): ?><div class="alert-success"><?= $success ?></div><?php endif; ?>

    <?php if ($rendicion): ?>
    <div class="rendicion-info">
        <h2><?= e($rendicion['titulo']) ?> (<?= e($rendicion['ssr_nombre']) ?>)</h2>
        <p><strong>Período:</strong> <?= date('M/Y', strtotime($rendicion['periodo_inicio'])) ?> - <?= date('M/Y', strtotime($rendicion['periodo_fin'])) ?></p>
        <p><strong>Estado:</strong> <span style="font-weight: bold; color: <?= $rendicion['estado'] == 'publicada' ? var('--success') : '#999' ?>;"><?= e(ucfirst($rendicion['estado'])) ?></span></p>
        <?php if($rendicion['pdf1']): ?>
            <a href="uploads/<?= e($rendicion['pdf1']) ?>" target="_blank"><i class="fas fa-file-pdf"></i> Ver PDF Principal</a>
        <?php endif; ?>
    </div>

    <?php if ($user['rol'] == 5): // Formulario para Socios ?>
    <div class="pregunta-form">
        <h3><i class="fas fa-question-circle"></i> Haz una pregunta</h3>
        <form method="POST">
            <input type="hidden" name="hacer_pregunta" value="1">
            <textarea name="pregunta" rows="3" placeholder="Escribe tu pregunta sobre esta rendición..." required maxlength="500"></textarea>
            <div style="text-align: right;">
                <button type="submit" class="btn btn-send"><i class="fas fa-paper-plane"></i> Enviar Pregunta</button>
            </div>
        </form>
    </div>
    <?php endif; ?>

    <h3>Discusión (<?= count($preguntas) ?> preguntas)</h3>
    <?php if (empty($preguntas)): ?>
        <p style="text-align:center;padding:40px;color:#666;background:white;border-radius:12px;">📭 No hay preguntas para esta rendición.</p>
    <?php endif; ?>

    <?php foreach($preguntas as $p): ?>
    <div class="pregunta" id="q<?= $p['id'] ?>">
        <h4><?= e($p['pregunta']) ?></h4>
        <div class="pregunta-meta">
            Preguntado por: <strong><?= e($p['socio_pregunta']) ?></strong> (<?= date('d/m/Y H:i', strtotime($p['created_at'])) ?>)
        </div>
        
        <?php if ($p['respondido']): ?>
            <div class="respuesta">
                <strong>Respuesta del administrador (<?= e($p['admin_respuesta']) ?>):</strong>
                <?= nl2br(e($p['respuesta'])) ?>
            </div>
        <?php else: ?>
            <div class="no-respuesta">
                ⏳ Esperando respuesta del administrador
            </div>
            
            <?php if($user['rol'] <= 2 && $can_admin_interact): // Solo Admin/Superadmin con permiso de SSR puede responder ?>
            <button class="btn btn-responder" onclick="toggleForm(<?= $p['id'] ?>)">✍️ Responder</button>
            <div class="respuesta-form" id="form<?= $p['id'] ?>" style="display:none;">
                <form method="POST">
                    <input type="hidden" name="pregunta_id" value="<?= $p['id'] ?>">
                    <input type="hidden" name="responder" value="1">
                    <textarea name="respuesta" rows="3" placeholder="Respuesta clara y precisa... (ej: 'Página 5 del PDF')" required maxlength="1000"></textarea>
                    <div style="margin-top:10px; text-align: right;">
                        <button type="submit" class="btn btn-responder">📤 Publicar Respuesta</button>
                        <button type="button" class="btn btn-back" onclick="toggleForm(<?= $p['id'] ?>)">❌ Cancelar</button>
                    </div>
                </form>
            </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
    <?php endforeach; ?>
    
    <?php else: // Si la rendicion no existe o el Admin SSR no tiene acceso ?>
    <div class="no-access" style="text-align:center; padding: 60px;">
        <h3>⚠️ Acceso Denegado</h3>
        <p style="color:var(--danger); font-size: 1.1em;"><?= e($access_error) ?></p>
        <p>Verifique sus permisos o asegúrese de que la Rendición exista.</p>
    </div>
    <?php endif; ?>

    <div style="text-align:center;margin-top:30px;">
        <a href="<?= $back_link ?>" class="btn btn-back"><i class="fas fa-arrow-left"></i> Volver a la Lista</a>
    </div>
</div>

<script>
function toggleForm(id) {
    var form = document.getElementById('form' + id);
    if (form.style.display === 'none') {
        form.style.display = 'block';
    } else {
        form.style.display = 'none';
    }
}
</script>
</body>
</html>