<?php
session_start();
// =========================================
// ✅ FIX 1: Mostrar todos los errores para evitar el Error 500 (Fatal Error)
ini_set('display_errors', 1);
error_reporting(E_ALL);
// =========================================

// Usa la configuración y conexión estándar
$config = include 'config.php';

// =========================================
// ✅ FIX 2: Bloque try-catch para manejar fallos en la conexión PDO
try {
    $dsn = "mysql:host={$config['db']['host']};dbname={$config['db']['dbname']};charset=utf8mb4";
    $pdo = new PDO($dsn, $config['db']['user'], $config['db']['pass'], [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
    ]);
} catch (\PDOException $e) {
    // Si la conexión falla (credenciales incorrectas, base de datos no existe),
    // mostramos el error de inmediato y detenemos la ejecución.
    die("
        <div style='background:#f8d7da;color:#721c24;padding:20px;border:1px solid #f5c6cb;border-radius:10px;margin:30px;font-family:sans-serif;'>
            <h2>❌ Error Fatal de Conexión a la Base de Datos</h2>
            <p><strong>Causa:</strong> Revise las credenciales en <code>config.php</code> o asegúrese que la base de datos existe.</p>
            <p><strong>Detalle Técnico:</strong> {$e->getMessage()}</p>
        </div>
    ");
}
// =========================================

$user = $_SESSION['user'] ?? null;
// Únicamente el Superadmin (Rol 1) puede gestionar los SSRs
if (!$user || $user['rol'] != 1) { 
    header('Location: index.php'); 
    exit; 
} 

$msg = '';
// ... (El resto del código PHP continúa desde aquí)
$edit_ssr = null;

// --- 1. LÓGICA DE MANEJO DE FORMULARIO (POST: Crear/Actualizar) ---
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['nombre'])) {
    $id = (int)($_POST['id'] ?? 0);
    $nombre = trim($_POST['nombre']);
    $descripcion = trim($_POST['descripcion'] ?? '');
    
    if (empty($nombre)) {
        $msg = "<div style='color:red;'>❌ El nombre del SSR es obligatorio.</div>";
    } else {
        // ACTUALIZAR
        if ($id > 0) {
            $stmt = $pdo->prepare("UPDATE ssr SET nombre = ?, descripcion = ? WHERE id = ?");
            $stmt->execute([$nombre, $descripcion, $id]);
            $msg = "<div style='color:green;'>✅ SSR actualizado con éxito (ID: $id).</div>";
        } 
        // CREAR
        else {
            $stmt = $pdo->prepare("INSERT INTO ssr (nombre, descripcion, activo) VALUES (?, ?, 1)");
            $stmt->execute([$nombre, $descripcion]);
            $msg = "<div style='color:green;'>✅ SSR creado con éxito (ID: " . $pdo->lastInsertId() . ").</div>";
        }
        header('Location: admin_ssr.php?msg=' . urlencode($msg)); 
        exit;
    }
}

// --- 2. LÓGICA DE MANEJO DE ACCIONES (GET: Editar/Activar/Desactivar) ---
if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    // Borrado Lógico: Desactivar para no romper las Foreign Keys
    $stmt = $pdo->prepare("UPDATE ssr SET activo = 0 WHERE id = ?"); 
    $stmt->execute([$id]);
    $msg = "<div style='color:orange;'>⚠️ SSR ID $id ha sido desactivado (Borrado Lógico).</div>";
} elseif (isset($_GET['activate'])) {
    $id = (int)$_GET['activate'];
    $stmt = $pdo->prepare("UPDATE ssr SET activo = 1 WHERE id = ?");
    $stmt->execute([$id]);
    $msg = "<div style='color:green;'>✅ SSR ID $id activado.</div>";
} elseif (isset($_GET['edit'])) {
    $id = (int)$_GET['edit'];
    $stmt = $pdo->prepare("SELECT * FROM ssr WHERE id = ?");
    $stmt->execute([$id]);
    $edit_ssr = $stmt->fetch();
} elseif (isset($_GET['msg'])) {
    $msg = urldecode($_GET['msg']);
}

// --- 3. CONSULTA PRINCIPAL (R: Leer/Listar) ---
$stmt = $pdo->query("SELECT id, nombre, descripcion, activo FROM ssr ORDER BY activo DESC, nombre");
$all_ssr = $stmt->fetchAll();

// Se define el título y la acción para el formulario
$form_title = $edit_ssr ? '✏️ Editar SSR (Empresa)' : '➕ Crear Nuevo SSR (Empresa)';
$button_text = $edit_ssr ? '💾 Actualizar SSR' : '➕ Crear SSR';
?>
<!DOCTYPE html>
<html>
<head>
    <title>🏢 Admin SSR (Empresas)</title>
    <meta charset="utf-8">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        /* Estilos base (puedes reutilizar los de admin_socios.php si son más limpios) */
        :root { --primary: #77216F; --success: #28a745; --danger: #dc3545; --warning: #ffc107; }
        body{font-family:'Segoe UI',Arial,sans-serif;background:#f8f9fa;padding:20px;min-height:100vh;}
        .container{max-width:1000px;margin:0 auto;}
        .header{background:linear-gradient(135deg,var(--primary),#5E2750);color:white;padding:30px;border-radius:15px;text-align:center;margin-bottom:20px;box-shadow:0 10px 30px rgba(119,33,111,0.3);}
        .form-container, table{background:white;padding:30px;border-radius:15px;margin:20px 0;box-shadow:0 10px 30px rgba(0,0,0,0.1);}
        .form-input, .form-textarea{padding:14px;border:2px solid #e9ecef;border-radius:10px;width:100%;font-size:16px;margin-bottom:15px;}
        .btn{display:inline-block;padding:12px 24px;background:var(--primary);color:white;text-decoration:none;border-radius:8px;font-weight:600;font-size:15px;transition:all 0.3s;margin:5px;border:none;cursor:pointer;}
        .btn-success{background:var(--success);} .btn-danger{background:var(--danger);} .btn-warning{background:var(--warning);color:#333;}
        table{width:100%;border-collapse:collapse;} th{padding:18px 12px;background:var(--primary);color:white;text-align:left;font-weight:600;}
        td{padding:16px 12px;border-bottom:1px solid #f1f3f4;vertical-align:middle;} tr:hover{background:#f8f9fa;}
        .status-active{background:#d4edda;color:#155724;padding:6px 12px;border-radius:20px;font-size:14px;font-weight:500;}
        .status-inactive{background:#f8d7da;color:#721c24;padding:6px 12px;border-radius:20px;font-size:14px;font-weight:500;}
        .action-btns a{margin: 0 3px;}
    </style>
</head>
<body>
<div class="container">
    <div class="header">
        <h1><i class="fas fa-building"></i> Administración de SSRs (Empresas)</h1>
        <p>Control de las entidades que agrupan a los socios.</p>
    </div>

    <?php if(!empty($msg)): ?>
        <div style="padding: 15px; margin: 20px 0; border-radius: 8px; font-weight: 600;"><?= $msg ?></div>
    <?php endif; ?>

    <div class="form-container">
        <h2><?= $form_title ?></h2>
        <form method="POST">
            <input type="hidden" name="id" value="<?= $edit_ssr['id'] ?? '' ?>">
            
            <label>Nombre del SSR *</label>
            <input class="form-input" type="text" name="nombre" value="<?= htmlspecialchars($edit_ssr['nombre'] ?? '') ?>" required>
            
            <label>Descripción / Comentarios</label>
            <textarea class="form-textarea" name="descripcion" rows="3"><?= htmlspecialchars($edit_ssr['descripcion'] ?? '') ?></textarea>

            <button type="submit" class="btn"><?= $button_text ?></button>
            <?php if($edit_ssr): ?>
                <a href="admin_ssr.php" class="btn" style="background:#6c757d;">❌ Cancelar Edición</a>
            <?php endif; ?>
            <a href="index.php?p=dashboard" class="btn btn-warning">← Dashboard</a>
        </form>
    </div>
    
    <div class="form-container">
        <h2>📋 Listado de SSRs</h2>
        <?php if($all_ssr): ?>
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Nombre</th>
                    <th>Descripción</th>
                    <th>Estado</th>
                    <th>Acciones</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach($all_ssr as $ssr): ?>
                <tr>
                    <td><strong><?= $ssr['id'] ?></strong></td>
                    <td><?= htmlspecialchars($ssr['nombre']) ?></td>
                    <td><?= htmlspecialchars($ssr['descripcion'] ?? '-') ?></td>
                    <td>
                        <span class="<?= $ssr['activo'] ? 'status-active' : 'status-inactive' ?>">
                            <?= $ssr['activo'] ? '🟢 Activo' : '🔴 Inactivo' ?>
                        </span>
                    </td>
                    <td class="action-btns">
                        <a href="?edit=<?= $ssr['id'] ?>" class="btn btn-warning" title="Editar"><i class="fas fa-edit"></i></a>
                        <?php if($ssr['activo']): ?>
                            <a href="?delete=<?= $ssr['id'] ?>" class="btn btn-danger" onclick="return confirm('¿Desactivar el SSR: <?= htmlspecialchars($ssr['nombre']) ?>?')" title="Desactivar (Borrado Lógico)"><i class="fas fa-trash"></i></a>
                        <?php else: ?>
                            <a href="?activate=<?= $ssr['id'] ?>" class="btn btn-success" title="Activar"><i class="fas fa-check"></i></a>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php else: ?>
        <p>Aún no hay SSRs creados.</p>
        <?php endif; ?>
    </div>
</div>
</body>
</html>