<?php
session_start();
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once 'config.php';
require_once 'rut.php';

function e($text) { return htmlspecialchars($text ?? '', ENT_QUOTES, 'UTF-8'); }
function user() { return $_SESSION['user'] ?? null; }

$user = user();

// Control de Acceso: Solo Rol 1 (Superadmin) y Rol 2 (Admin SSR)
if (!$user || $user['rol'] > 2) {
    header('Location: index.php');
    exit;
}

$config = include 'config.php';
$dsn = "mysql:host={$config['db']['host']};dbname={$config['db']['dbname']};charset=utf8mb4";
$pdo = new PDO($dsn, $config['db']['user'], $config['db']['pass'], [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
]);

$success = $error = '';
$ssr_filter_id = 0;

// ===============================================
// Lógica de Filtrado por SSR
// ===============================================

// Rol 2 (Admin SSR) solo ve SU SSR, forzamos el ID
if ($user['rol'] == 2) {
    $ssr_filter_id = $user['ssr_id'] ?? 0;
    if ($ssr_filter_id == 0) {
        die("Error: Su cuenta de administrador no está asignada a una SSR válida.");
    }
} 
// Rol 1 (Superadmin) puede filtrar por cualquier SSR, incluyendo 0 (Todas)
elseif ($user['rol'] == 1) {
    $ssr_filter_id = (int)($_GET['ssr'] ?? 0);
}

// Obtener el nombre para el encabezado/filtro
$ssr_nombre = 'Todas las SSR';
if ($ssr_filter_id > 0) {
    $stmt = $pdo->prepare("SELECT nombre FROM ssr WHERE id = ?");
    $stmt->execute([$ssr_filter_id]);
    $ssr_nombre = $stmt->fetchColumn() ?: 'SSR Desconocida';
}


// ===============================================
// PROCESAMIENTO DELETE
// ===============================================

if (isset($_GET['delete'])) {
    $id_to_delete = (int)$_GET['delete'];
    $stmt = $pdo->prepare("SELECT ssr_id, titulo FROM rendiciones WHERE id = ?");
    $stmt->execute([$id_to_delete]);
    $rendicion_to_delete = $stmt->fetch();

    if ($rendicion_to_delete) {
        // Validación de Permisos de Eliminación
        if ($user['rol'] == 1 || ($user['rol'] == 2 && $rendicion_to_delete['ssr_id'] == $user['ssr_id'])) {
            // Eliminar registros de foro
            $pdo->prepare("DELETE FROM preguntas_rendiciones WHERE rendicion_id = ?")->execute([$id_to_delete]);
            // Eliminar la rendición
            $pdo->prepare("DELETE FROM rendiciones WHERE id = ?")->execute([$id_to_delete]);
            $success = "✅ Rendición '<strong>".e($rendicion_to_delete['titulo'])."</strong>' eliminada exitosamente.";
        } else {
            $error = "❌ No tiene permisos para eliminar esta rendición.";
        }
    } else {
        $error = "❌ Rendición no encontrada.";
    }
    header("Location: rendiciones_admin.php?ssr={$ssr_filter_id}");
    exit;
}


// ===============================================
// CONSULTA SQL CON FILTRO
// ===============================================

$where_clause = '';
$params = [];

if ($ssr_filter_id > 0) {
    $where_clause = "WHERE r.ssr_id = ?";
    $params[] = $ssr_filter_id;
}

$sql = "SELECT r.*, s.nombre as ssr_nombre, u.nombre_completo as publicador
        FROM rendiciones r 
        JOIN ssr s ON r.ssr_id = s.id
        JOIN socios u ON r.publicado_por = u.id
        {$where_clause}
        ORDER BY r.fecha DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$rendiciones = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html>
<head>
    <title>📋 Administrar Rendiciones</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        :root {
            --primary: #77216F;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --info: #17a2b8;
        }
        body{font-family:'Segoe UI',Arial,sans-serif;background:#f8f9fa;padding:20px;min-height:100vh;}
        .container{max-width:1200px;margin:0 auto;}
        .header{background:linear-gradient(135deg,var(--primary),#5E2750);color:white;padding:30px;border-radius:15px;text-align:center;margin-bottom:30px;}
        .alert-error{background:#f8d7da;color:var(--danger);padding:15px;border-radius:8px;margin-bottom:20px;border-left:5px solid var(--danger);}
        .alert-success{background:#d4edda;color:var(--success);padding:15px;border-radius:8px;margin-bottom:20px;border-left:5px solid var(--success);}
        table{width:100%;border-collapse:collapse;background:white;border-radius:12px;overflow:hidden;box-shadow:0 4px 20px rgba(0,0,0,0.1);margin-top:20px;}
        th{background:linear-gradient(135deg,var(--primary) 0%,#5E2750 100%);color:white;padding:15px;text-align:left;font-weight:600;}
        td{padding:15px;border-bottom:1px solid #eee;}
        tr:hover { background: #f8f8f8; }
        .btn{padding:8px 12px;border-radius:6px;font-weight:500;cursor:pointer;transition:all 0.3s;text-decoration:none;display:inline-block;margin:2px;border:none;}
        .btn-primary{background:var(--primary);color:white;}
        .btn-primary:hover{background:#5E2750;}
        .btn-success{background:var(--success);color:white;}
        .btn-warning{background:var(--warning);color:#333;}
        .btn-danger{background:var(--danger);color:white;}
        .btn-foro{background:var(--info);color:white;}
        .btn-subir{background:#0dcaf0;color:#333;}
        .btn-back{background:#6c757d;color:white;}
        .status-publicada { color: var(--success); font-weight: bold; }
        .status-borrador { color: var(--warning); font-weight: bold; }
        .actions-group a { margin-right: 5px; }
    </style>
</head>
<body>
<div class="container">
    <div class="header">
        <h1><i class="fas fa-tasks"></i> Administración de Rendiciones</h1>
        <p>Usuario: **<?= e($user['nombre_completo']) ?>** (<?= $user['rol']==1?'Superadmin':($user['rol']==2?'Admin SSR':'') ?>)</p>
    </div>

    <div style="margin-bottom: 20px; display: flex; justify-content: space-between; align-items: center;">
        <h2>📋 Listado de Rendiciones - **<?= e($ssr_nombre) ?>**</h2>
        <a href="nueva_rendicion.php?ssr=<?= $ssr_filter_id > 0 ? $ssr_filter_id : '' ?>" class="btn btn-primary"><i class="fas fa-plus"></i> Nueva Rendición</a>
    </div>

    <?php if ($error): ?><div class="alert-error"><?= $error ?></div><?php endif; ?>
    <?php if ($success): ?><div class="alert-success"><?= $success ?></div><?php endif; ?>

    <?php if ($user['rol'] == 1): // Filtro solo para Superadmin ?>
    <div style="margin-bottom: 20px; padding: 15px; background: #f0f0f0; border-radius: 8px;">
        <label for="ssr_selector" style="font-weight: 600;">Filtrar por SSR:</label>
        <select id="ssr_selector" onchange="window.location.href='rendiciones_admin.php?ssr=' + this.value" style="padding: 8px; border-radius: 6px; border: 1px solid #ccc;">
            <option value="0" <?= $ssr_filter_id == 0 ? 'selected' : '' ?>>🌐 Todas las SSR</option>
            <?php 
            $stmt_all_ssr = $pdo->query("SELECT id, nombre FROM ssr WHERE activo = 1 ORDER BY nombre");
            while ($s = $stmt_all_ssr->fetch()):
            ?>
                <option value="<?= $s['id'] ?>" <?= $s['id'] == $ssr_filter_id ? 'selected' : '' ?>>
                    <?= e($s['nombre']) ?>
                </option>
            <?php endwhile; ?>
        </select>
    </div>
    <?php endif; ?>


    <?php if (empty($rendiciones)): ?>
    <div style="text-align:center;padding:40px;background:white;border-radius:12px;box-shadow:0 4px 20px rgba(0,0,0,0.05);">
        <h3>📭 No hay rendiciones disponibles.</h3>
        <p>Cree una nueva rendición para comenzar.</p>
        <a href="index.php?p=dashboard" class="btn btn-back" style="margin-top: 15px;">← Volver al Dashboard</a>
    </div>
    <?php else: ?>
    <table>
        <tr>
            <th>ID</th>
            <th>Título</th>
            <th>SSR</th>
            <th>Período</th>
            <th>Publicado Por</th>
            <th>Estado</th>
            <th>PDF</th>
            <th>Foro</th>
            <th>Acciones</th>
        </tr>
        <?php foreach($rendiciones as $r): ?>
        <tr>
            <td><strong>#<?= $r['id'] ?></strong></td>
            <td><?= e($r['titulo']) ?></td>
            <td><?= e($r['ssr_nombre']) ?></td>
            <td><?= date('M/Y', strtotime($r['periodo_inicio'])) ?> - <?= date('M/Y', strtotime($r['periodo_fin'])) ?></td>
            <td><?= e($r['publicador']) ?></td>
            <td class="status-<?= $r['estado'] ?>"><?= $r['estado'] ?></td>
            <td>
                <?php if($r['pdf1']): ?>
                    <a href="uploads/<?= e($r['pdf1']) ?>" target="_blank" class="btn btn-success">📄 Ver</a>
                <?php else: ?>
                    <span style="color:#999;">—</span>
                <?php endif; ?>
            </td>
            <td><a href="foro_rendicion.php?id=<?= $r['id'] ?>" class="btn btn-foro">🗣️ Foro</a></td>
            <td class="actions-group">
                <a href="subir_pdf.php?id=<?= $r['id'] ?>" class="btn btn-subir">📤 PDF</a>
                <a href="editar_rendicion.php?id=<?= $r['id'] ?>" class="btn btn-warning">✏️ Editar</a>
                <a href="?delete=<?= $r['id'] ?>&ssr=<?= $ssr_filter_id ?>" class="btn btn-danger" onclick="return confirm('¿Eliminar rendición #<?= $r['id'] ?> (<?= e($r['titulo']) ?>)? Esto eliminará también todas las preguntas asociadas.')">🗑️ Borrar</a>
            </td>
        </tr>
        <?php endforeach; ?>
    </table>
    <?php endif; ?>
    
    <div style="text-align:center;margin-top:30px;">
        <a href="index.php?p=dashboard" class="btn btn-back">← Volver al Dashboard</a>
    </div>
</div>
</body>
</html>