<?php
namespace Core\Controllers;

use Core\Classes\Auth;
use Core\Models\ContactabilidadCaso;
use Core\Models\ContactabilidadGestion;
use Core\Models\ContactabilidadAdjunto;
use Core\Models\ContactabilidadCliente;
use Core\Models\ContactabilidadAprobacion;

class ContactabilidadController {
    private $auth;
    private $casoModel;
    private $gestionModel;
    private $adjuntoModel;
    private $clienteModel;
    private $aprobacionModel;

    public function __construct() {
        $this->auth = new Auth();
        $this->casoModel = new ContactabilidadCaso();
        $this->gestionModel = new ContactabilidadGestion();
        $this->adjuntoModel = new ContactabilidadAdjunto();
        $this->clienteModel = new ContactabilidadCliente();
        $this->aprobacionModel = new ContactabilidadAprobacion();
    }

    // Listar casos
    public function index() {
        $this->checkAuth();
        $empresa_id = $_SESSION['empresa_id'];
        $casos = $this->casoModel->getByEmpresa($empresa_id);
        require __DIR__ . '/../../views/contactabilidad/index.php';
    }

    // Crear caso (form)
    public function create() {
        $this->checkAuth();
        require __DIR__ . '/../../views/contactabilidad/create.php';
    }

    // Guardar nuevo caso
    public function store() {
        $this->checkAuth();
        $data = [
            'empresa_id' => $_SESSION['empresa_id'],
            'cliente_id' => $_SESSION['user_id'],
            'tipo' => $_POST['tipo'] ?? 'reclamo',
            'estado' => 'recibido',
            'asunto' => $_POST['asunto'] ?? '',
            'descripcion' => $_POST['descripcion'] ?? '',
            'creado_por' => $_SESSION['user_id']
        ];
        if ($this->casoModel->create($data)) {
            header('Location: /contactabilidad');
            exit;
        }
        $error = 'Error al crear el caso';
        require __DIR__ . '/../../views/contactabilidad/create.php';
    }

    // Editar caso (form)
    public function edit($id) {
        $this->checkAuth();
        $caso = $this->casoModel->getById($id);
        $gestiones = $this->gestionModel->getByCaso($id);
        $adjuntos = $this->adjuntoModel->getByCaso($id);
        $aprobaciones = $this->aprobacionModel->getByCaso($id);
        if (!$caso) {
            header('Location: /contactabilidad');
            exit;
        }
        require __DIR__ . '/../../views/contactabilidad/edit.php';
    }

    // Actualizar caso
    public function update($id) {
        $this->checkAuth();
        $data = [
            'tipo' => $_POST['tipo'] ?? 'reclamo',
            'estado' => $_POST['estado'] ?? 'recibido',
            'asunto' => $_POST['asunto'] ?? '',
            'descripcion' => $_POST['descripcion'] ?? '',
            'fecha_cierre' => $_POST['fecha_cierre'] ?? null,
            'asignado_a' => $_POST['asignado_a'] ?? null,
        ];
        if ($this->casoModel->update($id, $data)) {
            header('Location: /contactabilidad');
            exit;
        }
        $error = 'Error al actualizar el caso';
        $caso = $this->casoModel->getById($id);
        require __DIR__ . '/../../views/contactabilidad/edit.php';
    }

    // Eliminar caso
    public function delete($id) {
        $this->checkAuth();
        $this->casoModel->delete($id);
        header('Location: /contactabilidad');
        exit;
    }

    // Agregar gestión a caso
    public function addGestion($caso_id) {
        $this->checkAuth();
        $data = [
            'caso_id' => $caso_id,
            'usuario_id' => $_SESSION['user_id'],
            'accion' => $_POST['accion'] ?? '',
        ];
        if ($this->gestionModel->create($data)) {
            header("Location: /contactabilidad/edit/$caso_id");
            exit;
        }
        $error = 'Error al agregar gestión';
        $this->edit($caso_id);
    }

    // Agregar adjunto a caso (suponiendo manejo básico sin upload)
    public function addAdjunto($caso_id) {
        $this->checkAuth();
        // Aquí debe implementarse lógica real de subida y almacenamiento
        $data = [
            'caso_id' => $caso_id,
            'archivo' => $_POST['archivo'] ?? 'archivo-ejemplo.pdf',
            'tipo_archivo' => $_POST['tipo_archivo'] ?? 'pdf',
        ];
        if ($this->adjuntoModel->create($data)) {
            header("Location: /contactabilidad/edit/$caso_id");
            exit;
        }
        $error = 'Error al agregar adjunto';
        $this->edit($caso_id);
    }

    // Agregar aprobación a caso
    public function addAprobacion($caso_id) {
        $this->checkAuth();
        $data = [
            'caso_id' => $caso_id,
            'aprobado_por' => $_SESSION['user_id'],
            'comentario' => $_POST['comentario'] ?? '',
        ];
        if ($this->aprobacionModel->create($data)) {
            header("Location: /contactabilidad/edit/$caso_id");
            exit;
        }
        $error = 'Error al agregar aprobación';
        $this->edit($caso_id);
    }

    private function checkAuth() {
        if (!$this->auth->check()) {
            header('Location: /login');
            exit;
        }
    }
}
