<?php
namespace Core\Controllers;

use Core\Classes\Auth;
use Core\Models\Usuario;

class UsuarioController {
    private $auth;
    private $model;

    public function __construct() {
        $this->auth = new Auth();
        $this->model = new Usuario();
    }

    // ============================================
    // AUTENTICACIÓN
    // ============================================

    /**
     * Muestra el formulario de login
     */
    public function loginForm() {
        // Usar constante BASE_PATH o ruta absoluta
        $vista = defined('BASE_PATH') ? BASE_PATH . '/views/auth/login.php' : __DIR__ . '/../../views/auth/login.php';
        
        if (!file_exists($vista)) {
            die("Vista no encontrada: $vista");
        }
        
        require $vista;
    }

    /**
     * Procesa el login
     */
    public function login() {
        $email = $_POST['email'] ?? null;
        $password = $_POST['password'] ?? null;

        // Validación básica
        if (!$email || !$password) {
            $error = 'Por favor ingrese email y contraseña';
            $vista = defined('BASE_PATH') ? BASE_PATH . '/views/auth/login.php' : __DIR__ . '/../../views/auth/login.php';
            require $vista;
            return;
        }

        // Buscar usuario por email
        $user = $this->model->getByEmail($email);
        
        // Verificar credenciales
        if ($user && password_verify($password, $user['password_hash'])) {
            // Login exitoso - Guardar datos en sesión
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['empresa_id'] = $user['empresa_id'];
            $_SESSION['user_role'] = $user['rol'];
            $_SESSION['user_email'] = $user['email'];
            $_SESSION['user_name'] = $user['nombres'] . ' ' . $user['apellidos'];
            
            // Redirigir al dashboard
            header('Location: /dashboard');
            exit;
        } else {
            // Credenciales inválidas
            $error = 'Credenciales inválidas. Por favor verifique su email y contraseña.';
            $vista = defined('BASE_PATH') ? BASE_PATH . '/views/auth/login.php' : __DIR__ . '/../../views/auth/login.php';
            require $vista;
        }
    }

    /**
     * Cierra sesión
     */
    public function logout() {
        $this->auth->logout();
        header('Location: /login');
        exit;
    }

    // ============================================
    // GESTIÓN DE USUARIOS
    // ============================================

    /**
     * Lista todos los usuarios
     */
    public function index() {
        $this->checkAuth();
        $usuarios = $this->model->getAll();
        $vista = defined('BASE_PATH') ? BASE_PATH . '/views/usuario/index.php' : __DIR__ . '/../../views/usuario/index.php';
        require $vista;
    }

    /**
     * Muestra formulario para crear usuario
     */
    public function create() {
        $this->checkAuth();
        $vista = defined('BASE_PATH') ? BASE_PATH . '/views/usuario/create.php' : __DIR__ . '/../../views/usuario/create.php';
        require $vista;
    }

    /**
     * Guarda un nuevo usuario
     */
    public function store() {
        $this->checkAuth();
        
        $data = [
            'empresa_id' => $_SESSION['empresa_id'],
            'rut' => $_POST['rut'] ?? '',
            'nombres' => $_POST['nombres'] ?? '',
            'apellidos' => $_POST['apellidos'] ?? '',
            'email' => $_POST['email'] ?? '',
            'password' => $_POST['password'] ?? '',
            'telefono' => $_POST['telefono'] ?? '',
            'rol' => $_POST['rol'] ?? 'cliente',
            'estado' => 'activo'
        ];
        
        if ($this->model->create($data)) {
            header('Location: /usuario');
            exit;
        }
        
        $error = 'Error al crear usuario';
        $vista = defined('BASE_PATH') ? BASE_PATH . '/views/usuario/create.php' : __DIR__ . '/../../views/usuario/create.php';
        require $vista;
    }

    /**
     * Muestra formulario para editar usuario
     */
    public function edit($id) {
        $this->checkAuth();
        
        $usuario = $this->model->getById($id);
        if (!$usuario) {
            header('Location: /usuario');
            exit;
        }
        
        $vista = defined('BASE_PATH') ? BASE_PATH . '/views/usuario/edit.php' : __DIR__ . '/../../views/usuario/edit.php';
        require $vista;
    }

    /**
     * Actualiza un usuario
     */
    public function update($id) {
        $this->checkAuth();
        
        $data = [
            'rut' => $_POST['rut'] ?? '',
            'nombres' => $_POST['nombres'] ?? '',
            'apellidos' => $_POST['apellidos'] ?? '',
            'email' => $_POST['email'] ?? '',
            'telefono' => $_POST['telefono'] ?? '',
            'rol' => $_POST['rol'] ?? 'cliente',
            'estado' => $_POST['estado'] ?? 'activo'
        ];
        
        // Actualizar contraseña solo si se proporciona
        if (!empty($_POST['password'])) {
            $data['password'] = $_POST['password'];
        }
        
        if ($this->model->update($id, $data)) {
            header('Location: /usuario');
            exit;
        }
        
        $error = 'Error al actualizar usuario';
        $usuario = $this->model->getById($id);
        $vista = defined('BASE_PATH') ? BASE_PATH . '/views/usuario/edit.php' : __DIR__ . '/../../views/usuario/edit.php';
        require $vista;
    }

    /**
     * Elimina un usuario (soft delete)
     */
    public function delete($id) {
        $this->checkAuth();
        $this->model->delete($id);
        header('Location: /usuario');
        exit;
    }

    // ============================================
    // MÉTODOS AUXILIARES
    // ============================================

    /**
     * Verifica si el usuario está autenticado
     */
    private function checkAuth() {
        if (!$this->auth->check()) {
            header('Location: /login');
            exit;
        }
    }
}
