<?php
namespace Core\Classes;

/**
 * Sistema de Control de Acceso Basado en Roles (RBAC)
 * Gestiona permisos para 6 roles: superadmin, adminempresa, directorio, ejecutivo, supervisor, cliente
 */
class Permission {
    
    // Definición completa de permisos por rol según documentación SG-Reclamos
    private static $permissions = [
        'superadmin' => [
            // Gestión del sistema
            'crear_empresas',
            'gestionar_usuarios_global',
            'ver_todas_empresas',
            'cambiar_empresa',
            'configurar_sistema',
            
            // Módulos generales
            'ver_dashboard',
            'ver_todos_casos',
            'crear_caso',
            'asignar_caso',
            'responder_caso',
            'eliminar_caso',
            
            // Reportes y auditoría
            'ver_reportes',
            'exportar_siss',
            'ver_logs',
            'backup_sistema',
            
            // Gestión de datos
            'cargar_clientes',
            'gestionar_bancos',
            'gestionar_tesoreria',
            'gestionar_proveedores',
            'gestionar_documentos'
        ],
        
        'adminempresa' => [
            // Gestión de su empresa
            'gestionar_usuarios',
            'configurar_ssr',
            'cargar_clientes',
            
            // Dashboard y casos
            'ver_dashboard',
            'ver_todos_casos',
            'crear_caso',
            'asignar_caso',
            'responder_caso',
            'aprobar_respuesta',
            'eliminar_caso',
            
            // Módulos financieros
            'gestionar_bancos',
            'gestionar_tesoreria',
            'gestionar_proveedores',
            
            // Reportes
            'ver_reportes',
            'exportar_siss',
            'ver_logs',
            
            // Gestión de datos
            'gestionar_clientes',
            'gestionar_documentos',
            'gestionar_categorias'
        ],
        
        'directorio' => [
            // Dashboard y visualización
            'ver_dashboard',
            'ver_todos_casos',
            
            // Gestión de casos
            'crear_caso',
            'asignar_caso',
            'responder_caso',
            'aprobar_respuesta',
            
            // Reportes ejecutivos
            'ver_reportes',
            'exportar_siss',
            
            // Consulta de datos
            'ver_clientes',
            'ver_bancos',
            'ver_tesoreria'
        ],
        
        'supervisor' => [
            // Dashboard y casos
            'ver_dashboard',
            'ver_todos_casos',
            'crear_caso',
            'asignar_caso',
            'responder_caso',
            'aprobar_respuesta',
            
            // Reportes
            'ver_reportes',
            'exportar_siss',
            
            // Consulta
            'ver_clientes',
            'ver_seguimientos'
        ],
        
        'ejecutivo' => [
            // Solo casos asignados
            'ver_mis_casos',
            'crear_caso',
            'responder_caso',
            'ver_seguimientos',
            
            // Consulta básica
            'ver_clientes',
            'buscar_clientes'
        ],
        
        'cliente' => [
            // Portal público
            'ver_mis_casos',
            'crear_caso_portal',
            'ver_mis_respuestas',
            'ver_mi_perfil',
            'actualizar_mi_perfil'
        ]
    ];
    
    /**
     * Verifica si el usuario actual tiene un permiso específico
     */
    public static function can($permission) {
        if (!isset($_SESSION['user_role'])) {
            return false;
        }
        
        $role = $_SESSION['user_role'];
        
        // Superadmin tiene acceso total
        if ($role === 'superadmin') {
            return true;
        }
        
        // Verificar permiso específico del rol
        return isset(self::$permissions[$role]) && 
               in_array($permission, self::$permissions[$role]);
    }
    
    /**
     * Verifica múltiples permisos (OR - al menos uno)
     */
    public static function canAny(array $permissions) {
        foreach ($permissions as $permission) {
            if (self::can($permission)) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Verifica múltiples permisos (AND - todos)
     */
    public static function canAll(array $permissions) {
        foreach ($permissions as $permission) {
            if (!self::can($permission)) {
                return false;
            }
        }
        return true;
    }
    
    /**
     * Redirige si no tiene permiso (403 Forbidden)
     */
    public static function requirePermission($permission) {
        if (!self::can($permission)) {
            self::showAccessDenied();
        }
    }
    
    /**
     * Muestra página de acceso denegado
     */
    private static function showAccessDenied() {
        header('HTTP/1.1 403 Forbidden');
        echo "<!DOCTYPE html>";
        echo "<html lang='es'>";
        echo "<head>";
        echo "<meta charset='UTF-8'>";
        echo "<meta name='viewport' content='width=device-width, initial-scale=1.0'>";
        echo "<title>Acceso Denegado - SG-SSR</title>";
        echo "<link rel='stylesheet' href='https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css'>";
        echo "<style>";
        echo "body { font-family: 'Segoe UI', Tahoma, sans-serif; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); display: flex; align-items: center; justify-content: center; min-height: 100vh; margin: 0; }";
        echo ".error-box { background: white; padding: 60px 40px; border-radius: 16px; box-shadow: 0 20px 60px rgba(0,0,0,0.3); text-align: center; max-width: 500px; }";
        echo ".error-icon { font-size: 5rem; color: #dc3545; margin-bottom: 30px; }";
        echo "h1 { color: #2c3e50; margin-bottom: 20px; font-size: 2rem; }";
        echo "p { color: #6c757d; margin-bottom: 30px; font-size: 1.1rem; line-height: 1.6; }";
        echo ".btn { display: inline-block; padding: 12px 30px; background: #667eea; color: white; text-decoration: none; border-radius: 8px; font-weight: 600; transition: all 0.3s; }";
        echo ".btn:hover { background: #5568d3; transform: translateY(-2px); box-shadow: 0 10px 20px rgba(102,126,234,0.3); }";
        echo ".user-info { background: #f8f9fa; padding: 15px; border-radius: 8px; margin-top: 20px; font-size: 0.9rem; color: #495057; }";
        echo "</style>";
        echo "</head>";
        echo "<body>";
        echo "<div class='error-box'>";
        echo "<div class='error-icon'><i class='fas fa-ban'></i></div>";
        echo "<h1>🚫 Acceso Denegado</h1>";
        echo "<p>No tienes permisos suficientes para acceder a este recurso.</p>";
        echo "<div class='user-info'>";
        echo "<strong>Usuario:</strong> " . htmlspecialchars($_SESSION['user_name'] ?? 'Desconocido') . "<br>";
        echo "<strong>Rol:</strong> " . htmlspecialchars(self::getRoleName($_SESSION['user_role'] ?? '')) . "<br>";
        echo "<strong>Empresa:</strong> " . htmlspecialchars($_SESSION['empresa_nombre'] ?? 'N/A');
        echo "</div>";
        echo "<a href='/dashboard' class='btn'><i class='fas fa-home'></i> Volver al Dashboard</a>";
        echo "</div>";
        echo "</body>";
        echo "</html>";
        exit;
    }
    
    /**
     * Obtiene el nombre traducido del rol
     */
    public static function getRoleName($role) {
        $roles = [
            'superadmin' => 'Super Administrador',
            'adminempresa' => 'Administrador SSR',
            'directorio' => 'Miembro Directorio',
            'ejecutivo' => 'Ejecutivo',
            'supervisor' => 'Supervisor',
            'cliente' => 'Cliente'
        ];
        
        return $roles[$role] ?? 'Usuario';
    }
    
    /**
     * Obtiene el color del badge según el rol
     */
    public static function getRoleBadgeColor($role) {
        $colors = [
            'superadmin' => '#dc3545',      // Rojo
            'adminempresa' => '#007bff',    // Azul
            'directorio' => '#6f42c1',      // Púrpura
            'supervisor' => '#17a2b8',      // Cyan
            'ejecutivo' => '#28a745',       // Verde
            'cliente' => '#ffc107'          // Amarillo
        ];
        
        return $colors[$role] ?? '#6c757d';
    }
    
    /**
     * Obtiene todos los roles disponibles
     */
    public static function getAllRoles() {
        return [
            'superadmin' => 'Super Administrador',
            'adminempresa' => 'Administrador SSR',
            'directorio' => 'Miembro Directorio',
            'supervisor' => 'Supervisor',
            'ejecutivo' => 'Ejecutivo',
            'cliente' => 'Cliente'
        ];
    }
    
    /**
     * Verifica si puede ver el dashboard completo
     */
    public static function canViewDashboard() {
        return self::can('ver_dashboard');
    }
    
    /**
     * Verifica si puede gestionar usuarios
     */
    public static function canManageUsers() {
        return self::canAny(['gestionar_usuarios', 'gestionar_usuarios_global']);
    }
    
    /**
     * Verifica si puede ver todos los casos o solo los suyos
     */
    public static function canViewAllCases() {
        return self::can('ver_todos_casos');
    }
    
    /**
     * Verifica si puede asignar casos
     */
    public static function canAssignCases() {
        return self::can('asignar_caso');
    }
    
    /**
     * Verifica si puede aprobar respuestas
     */
    public static function canApproveResponses() {
        return self::can('aprobar_respuesta');
    }
    
    /**
     * Verifica si puede gestionar empresas (solo superadmin)
     */
    public static function canManageCompanies() {
        return self::can('crear_empresas');
    }
    
    /**
     * Verifica si puede exportar reportes
     */
    public static function canExport() {
        return self::can('exportar_siss');
    }
    
    /**
     * Verifica si puede gestionar módulos financieros
     */
    public static function canManageFinance() {
        return self::canAny(['gestionar_bancos', 'gestionar_tesoreria']);
    }
}
