<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Definir rutas base
define('BASE_PATH', __DIR__);
define('VIEWS_PATH', __DIR__ . '/views');

// Cargar archivos base manualmente
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/core/classes/Database.php';
require_once __DIR__ . '/core/classes/Auth.php';
require_once __DIR__ . '/core/classes/Session.php';
require_once __DIR__ . '/core/classes/Router.php';

// Autoload DEFINITIVO - Todas las carpetas en minúsculas
spl_autoload_register(function ($class) {
    if (strpos($class, 'Core\\') !== 0) {
        return;
    }
    
    $classWithoutPrefix = substr($class, 5);
    $parts = explode('\\', $classWithoutPrefix);
    
    if (count($parts) > 0) {
        $parts[0] = strtolower($parts[0]);
    }
    
    $filePath = __DIR__ . '/core/' . implode('/', $parts) . '.php';
    
    if (file_exists($filePath)) {
        require_once $filePath;
    } else {
        error_log("Autoload: No se encontró el archivo $filePath para la clase $class");
    }
});

// Iniciar sesión
\Core\Classes\Session::start();

$router = new \Core\Classes\Router();

// ============================================
// FUNCIÓN AUXILIAR PARA CARGAR CONTROLADORES
// ============================================

function loadController($controllerName, $method = 'index', ...$params) {
    if (!isset($_SESSION['user_id'])) {
        header('Location: /login');
        exit;
    }
    
    $controllerFile = BASE_PATH . "/core/controllers/{$controllerName}.php";
    
    if (file_exists($controllerFile)) {
        try {
            require_once $controllerFile;
            $controllerClass = "\\Core\\Controllers\\{$controllerName}";
            
            if (class_exists($controllerClass)) {
                $controller = new $controllerClass();
                if (method_exists($controller, $method)) {
                    // Llamar al método con los parámetros usando call_user_func_array
                    call_user_func_array([$controller, $method], $params);
                } else {
                    echo "<h1>Error: Método no encontrado</h1>";
                    echo "<p>Método: {$controllerClass}::{$method}()</p>";
                    echo "<p><a href='/dashboard'>Volver al Dashboard</a></p>";
                }
            } else {
                echo "<h1>Error: Clase no encontrada</h1>";
                echo "<p>Clase esperada: {$controllerClass}</p>";
                echo "<p><a href='/dashboard'>Volver al Dashboard</a></p>";
            }
        } catch (\Throwable $e) {
            echo "<!DOCTYPE html>";
            echo "<html lang='es'>";
            echo "<head>";
            echo "<meta charset='UTF-8'>";
            echo "<meta name='viewport' content='width=device-width, initial-scale=1.0'>";
            echo "<title>Error - SG-SSR</title>";
            echo "<style>";
            echo "body { font-family: 'Segoe UI', Tahoma, sans-serif; background: #f5f7fa; padding: 40px; }";
            echo ".error-container { max-width: 800px; margin: 0 auto; background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }";
            echo "h1 { color: #dc3545; margin-bottom: 20px; }";
            echo "p { color: #495057; margin: 10px 0; }";
            echo "details { margin-top: 20px; padding: 15px; background: #f8f9fa; border-radius: 4px; }";
            echo "pre { background: #343a40; color: #f8f9fa; padding: 15px; border-radius: 4px; overflow-x: auto; }";
            echo ".btn { display: inline-block; padding: 10px 20px; background: #667eea; color: white; text-decoration: none; border-radius: 6px; margin-top: 20px; }";
            echo ".btn:hover { background: #5568d3; }";
            echo "</style>";
            echo "</head>";
            echo "<body>";
            echo "<div class='error-container'>";
            echo "<h1>❌ Error al cargar el módulo</h1>";
            echo "<p><strong>Error:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
            echo "<p><strong>Archivo:</strong> " . htmlspecialchars($e->getFile()) . "</p>";
            echo "<p><strong>Línea:</strong> " . $e->getLine() . "</p>";
            echo "<details><summary>Ver detalles técnicos</summary><pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre></details>";
            echo "<a href='/dashboard' class='btn'>Volver al Dashboard</a>";
            echo "</div>";
            echo "</body>";
            echo "</html>";
        }
    } else {
        echo "<h1>Error: Controlador no encontrado</h1>";
        echo "<p>Archivo esperado: {$controllerFile}</p>";
        echo "<p><a href='/dashboard'>Volver al Dashboard</a></p>";
    }
}

// ============================================
// AUTENTICACIÓN
// ============================================

$router->add('GET', '/', function () {
    header('Location: /login');
    exit;
});

$router->add('GET', '/login', function() {
    if (isset($_SESSION['user_id'])) {
        header('Location: /dashboard');
        exit;
    }
    require BASE_PATH . '/views/auth/login.php';
});

$router->add('POST', '/login', function() {
    $email = $_POST['email'] ?? null;
    $password = $_POST['password'] ?? null;

    if (!$email || !$password) {
        $error = 'Por favor ingrese email y contraseña';
        require BASE_PATH . '/views/auth/login.php';
        return;
    }

    try {
        $db = \Core\Classes\Database::getInstance()->getConnection();
        $stmt = $db->prepare("SELECT * FROM usuarios WHERE email = ? AND estado = 'activo'");
        $stmt->execute([$email]);
        $user = $stmt->fetch(\PDO::FETCH_ASSOC);
        
        if ($user && password_verify($password, $user['password_hash'])) {
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['empresa_id'] = $user['empresa_id'];
            $_SESSION['user_role'] = $user['rol'];
            $_SESSION['user_email'] = $user['email'];
            $_SESSION['user_name'] = $user['nombres'] . ' ' . $user['apellidos'];
            
            header('Location: /dashboard');
            exit;
        } else {
            $error = 'Credenciales inválidas';
        }
    } catch (\Exception $e) {
        $error = 'Error del sistema: ' . $e->getMessage();
    }
    
    require BASE_PATH . '/views/auth/login.php';
});

$router->add('GET', '/logout', function() {
    session_unset();
    session_destroy();
    header('Location: /login');
    exit;
});

// ============================================
// DASHBOARD
// ============================================

$router->add('GET', '/dashboard', function () {
    if (!isset($_SESSION['user_id'])) {
        header('Location: /login');
        exit;
    }
    require BASE_PATH . '/views/dashboard/index.php';
});

// ============================================
// MÓDULOS - CLIENTES
// ============================================

$router->add('GET', '/cliente', function() { 
    loadController('ClienteController', 'index'); 
});

$router->add('GET', '/cliente/create', function() { 
    loadController('ClienteController', 'create'); 
});

$router->add('POST', '/cliente/store', function() { 
    loadController('ClienteController', 'store'); 
});

$router->add('GET', '/cliente/edit/{id}', function($id) { 
    loadController('ClienteController', 'edit', $id); 
});

$router->add('POST', '/cliente/update/{id}', function($id) { 
    loadController('ClienteController', 'update', $id); 
});

$router->add('GET', '/cliente/delete/{id}', function($id) { 
    loadController('ClienteController', 'delete', $id); 
});

// ============================================
// MÓDULOS - EMPRESAS
// ============================================

$router->add('GET', '/empresa', function() { 
    loadController('EmpresaController', 'index'); 
});

$router->add('GET', '/empresa/create', function() { 
    loadController('EmpresaController', 'create'); 
});

$router->add('POST', '/empresa/store', function() { 
    loadController('EmpresaController', 'store'); 
});

$router->add('GET', '/empresa/edit/{id}', function($id) { 
    loadController('EmpresaController', 'edit', $id); 
});

$router->add('POST', '/empresa/update/{id}', function($id) { 
    loadController('EmpresaController', 'update', $id); 
});

$router->add('GET', '/empresa/delete/{id}', function($id) { 
    loadController('EmpresaController', 'delete', $id); 
});

// ============================================
// MÓDULOS - CASOS
// ============================================

$router->add('GET', '/caso', function() { 
    loadController('CasoController', 'index'); 
});

$router->add('GET', '/caso/create', function() { 
    loadController('CasoController', 'create'); 
});

$router->add('POST', '/caso/store', function() { 
    loadController('CasoController', 'store'); 
});

$router->add('GET', '/caso/edit/{id}', function($id) { 
    loadController('CasoController', 'edit', $id); 
});

$router->add('POST', '/caso/update/{id}', function($id) { 
    loadController('CasoController', 'update', $id); 
});

$router->add('GET', '/caso/delete/{id}', function($id) { 
    loadController('CasoController', 'delete', $id); 
});

// ============================================
// MÓDULOS - SEGUIMIENTO
// ============================================

$router->add('GET', '/seguimiento', function() { 
    loadController('SeguimientoController', 'index'); 
});

$router->add('GET', '/seguimiento/create', function() { 
    loadController('SeguimientoController', 'create'); 
});

$router->add('POST', '/seguimiento/store', function() { 
    loadController('SeguimientoController', 'store'); 
});

$router->add('GET', '/seguimiento/edit/{id}', function($id) { 
    loadController('SeguimientoController', 'edit', $id); 
});

$router->add('POST', '/seguimiento/update/{id}', function($id) { 
    loadController('SeguimientoController', 'update', $id); 
});

$router->add('GET', '/seguimiento/delete/{id}', function($id) { 
    loadController('SeguimientoController', 'delete', $id); 
});

// ============================================
// MÓDULOS - USUARIOS
// ============================================

$router->add('GET', '/usuario', function() { 
    loadController('UsuarioController', 'index'); 
});

$router->add('GET', '/usuario/create', function() { 
    loadController('UsuarioController', 'create'); 
});

$router->add('POST', '/usuario/store', function() { 
    loadController('UsuarioController', 'store'); 
});

$router->add('GET', '/usuario/edit/{id}', function($id) { 
    loadController('UsuarioController', 'edit', $id); 
});

$router->add('POST', '/usuario/update/{id}', function($id) { 
    loadController('UsuarioController', 'update', $id); 
});

$router->add('GET', '/usuario/delete/{id}', function($id) { 
    loadController('UsuarioController', 'delete', $id); 
});

// ============================================
// MÓDULOS - CATEGORÍAS
// ============================================

$router->add('GET', '/categoria', function() { 
    loadController('CategoriaController', 'index'); 
});

$router->add('GET', '/categoria/create', function() { 
    loadController('CategoriaController', 'create'); 
});

$router->add('POST', '/categoria/store', function() { 
    loadController('CategoriaController', 'store'); 
});

$router->add('GET', '/categoria/edit/{id}', function($id) { 
    loadController('CategoriaController', 'edit', $id); 
});

$router->add('POST', '/categoria/update/{id}', function($id) { 
    loadController('CategoriaController', 'update', $id); 
});

$router->add('GET', '/categoria/delete/{id}', function($id) { 
    loadController('CategoriaController', 'delete', $id); 
});

// ============================================
// MÓDULOS - CONTACTABILIDAD
// ============================================

$router->add('GET', '/contactabilidad', function() { 
    loadController('ContactabilidadController', 'index'); 
});

$router->add('GET', '/contactabilidad/create', function() { 
    loadController('ContactabilidadController', 'create'); 
});

$router->add('POST', '/contactabilidad/store', function() { 
    loadController('ContactabilidadController', 'store'); 
});

$router->add('GET', '/contactabilidad/edit/{id}', function($id) { 
    loadController('ContactabilidadController', 'edit', $id); 
});

$router->add('POST', '/contactabilidad/update/{id}', function($id) { 
    loadController('ContactabilidadController', 'update', $id); 
});

$router->add('GET', '/contactabilidad/delete/{id}', function($id) { 
    loadController('ContactabilidadController', 'delete', $id); 
});

// ============================================
// MÓDULOS - BANCOS
// ============================================

$router->add('GET', '/banco', function() { 
    loadController('BancoController', 'index'); 
});

$router->add('GET', '/banco/create', function() { 
    loadController('BancoController', 'create'); 
});

$router->add('POST', '/banco/store', function() { 
    loadController('BancoController', 'store'); 
});

$router->add('GET', '/banco/edit/{id}', function($id) { 
    loadController('BancoController', 'edit', $id); 
});

$router->add('POST', '/banco/update/{id}', function($id) { 
    loadController('BancoController', 'update', $id); 
});

$router->add('GET', '/banco/delete/{id}', function($id) { 
    loadController('BancoController', 'delete', $id); 
});

// ============================================
// MÓDULOS - TESORERÍA
// ============================================

$router->add('GET', '/tesoreria', function() { 
    loadController('TesoreriaController', 'index'); 
});

$router->add('GET', '/tesoreria/create', function() { 
    loadController('TesoreriaController', 'create'); 
});

$router->add('POST', '/tesoreria/store', function() { 
    loadController('TesoreriaController', 'store'); 
});

$router->add('GET', '/tesoreria/edit/{id}', function($id) { 
    loadController('TesoreriaController', 'edit', $id); 
});

$router->add('POST', '/tesoreria/update/{id}', function($id) { 
    loadController('TesoreriaController', 'update', $id); 
});

$router->add('GET', '/tesoreria/delete/{id}', function($id) { 
    loadController('TesoreriaController', 'delete', $id); 
});

// ============================================
// MÓDULOS - PROVEEDORES
// ============================================

$router->add('GET', '/proveedor', function() { 
    loadController('ProveedorController', 'index'); 
});

$router->add('GET', '/proveedor/create', function() { 
    loadController('ProveedorController', 'create'); 
});

$router->add('POST', '/proveedor/store', function() { 
    loadController('ProveedorController', 'store'); 
});

$router->add('GET', '/proveedor/edit/{id}', function($id) { 
    loadController('ProveedorController', 'edit', $id); 
});

$router->add('POST', '/proveedor/update/{id}', function($id) { 
    loadController('ProveedorController', 'update', $id); 
});

$router->add('GET', '/proveedor/delete/{id}', function($id) { 
    loadController('ProveedorController', 'delete', $id); 
});

// ============================================
// MÓDULOS - DOCUMENTOS Y REPORTES
// ============================================

$router->add('GET', '/documento', function() { 
    loadController('DocumentoController', 'index'); 
});

$router->add('GET', '/documento/create', function() { 
    loadController('DocumentoController', 'create'); 
});

$router->add('POST', '/documento/store', function() { 
    loadController('DocumentoController', 'store'); 
});

$router->add('GET', '/documento/edit/{id}', function($id) { 
    loadController('DocumentoController', 'edit', $id); 
});

$router->add('POST', '/documento/update/{id}', function($id) { 
    loadController('DocumentoController', 'update', $id); 
});

$router->add('GET', '/documento/delete/{id}', function($id) { 
    loadController('DocumentoController', 'delete', $id); 
});

$router->add('GET', '/reporte', function() { 
    loadController('ReporteController', 'index'); 
});

$router->add('GET', '/log', function() { 
    loadController('LogController', 'index'); 
});

// ============================================
// MÓDULOS - OTROS
// ============================================

$router->add('GET', '/portal', function() { 
    loadController('PortalClienteController', 'index'); 
});

// Respuesta - Deshabilitado (se gestiona desde casos)
// $router->add('GET', '/respuesta', function() { loadController('RespuestaController', 'index'); });

// ============================================
// DISPATCH
// ============================================

$method = $_SERVER['REQUEST_METHOD'];
$path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$router->dispatch($method, $path);
